<?php

namespace App\Http\Controllers;

use App\Models\Deposit;
use App\Models\UserLedger;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class DepositController extends Controller
{
    // Step 1: Select Payment Method
    public function recharge()
    {
        $methods = PaymentMethod::where('status', 'active')->get();
        return view('app.main.deposit.index', compact('methods'));
    }

    // Step 2: Redirect based on type
    public function recharge_redirect($method_id)
    {
        $method = PaymentMethod::findOrFail($method_id);

        if ($method->type === 'usdt') {
            return redirect()->route('recharge.usdt', $method->id);
        } else {
            return redirect()->route('recharge.amount', $method->id);
        }
    }

    // Step 3a: Normal deposit page (wallet/bank)
    public function recharge_amount($method_id)
    {
        $method = PaymentMethod::findOrFail($method_id);
        return view('app.main.deposit.amount', compact('method'));
    }

    // Step 3b: USDT deposit page
    public function recharge_usdt($method_id)
    {
        $method = PaymentMethod::findOrFail($method_id);
        return view('app.main.deposit.usdt', compact('method'));
    }

    // Step 4: Confirm Deposit (common for both)
    public function recharge_confirm_submit(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric',
            'method_id' => 'required|exists:payment_methods,id',
            'transaction_id' => 'required|string',
        ]);

        $method = PaymentMethod::findOrFail($request->method_id);

        // ✅ Check min/max limits
        if ($request->amount < $method->minimum || $request->amount > $method->maximum) {
            return back()->withErrors("Amount must be between {$method->minimum} and {$method->maximum}");
        }

        // ✅ Create deposit (table ke mutabiq)
        $deposit = Deposit::create([
            'user_id' => Auth::id(),
            'method_name' => $method->channel,
            'address' => $method->address,
            'order_id' => Str::upper(Str::random(12)),
            'transaction_id' => $request->transaction_id,
            'amount' => $request->amount,
            'final_amount' => $request->amount,
            'date' => now()->format('Y-m-d H:i:s'),
            'status' => 'pending',
        ]);

        // ✅ Ledger entry (agar table me hai)
        UserLedger::create([
            'user_id' => Auth::id(),
            'reason' => 'user_deposit',
            'perticulation' => 'User deposit via ' . $method->channel,
            'amount' => $request->amount,
            'credit' => $request->amount,
            'debit' => 0,
            'status' => 'pending',
            'date' => now()->format('Y-m-d'),
        ]);

        return redirect()->route('recharge.success');
    }

    // Step 5: Success Page
    public function recharge_success()
    {
        return view('app.main.deposit.recharge_success');
    }
}
