<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WealthPlan;
use App\Models\UserWealth;
use App\Models\User;
use Carbon\Carbon;

class AdminWealthController extends Controller
{
    // Show all wealth plans
    public function plans()
    {
        $plans = WealthPlan::all();
        return view('admin.wealth.plans', compact('plans'));
    }

    // Add new plan
    public function storePlan(Request $request)
    {
        $request->validate([
            'name'=>'required|string|max:255',
            'min_amount'=>'required|numeric|min:1',
            'max_amount'=>'required|numeric|gte:min_amount',
            'days'=>'required|integer|min:1',
            'profit_percent'=>'required|numeric|min:0'
        ]);

        WealthPlan::create([
            'name' => $request->name,
            'min_amount' => $request->min_amount,
            'max_amount' => $request->max_amount,
            'days' => $request->days,
            'profit_percent' => $request->profit_percent,
            'status' => 1, // default active
        ]);

        return back()->with('success','Plan added successfully!');
    }

    // Update existing plan
    public function updatePlan(Request $request, $id)
    {
        $request->validate([
            'name'=>'required|string|max:255',
            'min_amount'=>'required|numeric|min:1',
            'max_amount'=>'required|numeric|gte:min_amount',
            'days'=>'required|integer|min:1',
            'profit_percent'=>'required|numeric|min:0'
        ]);

        $plan = WealthPlan::findOrFail($id);
        $plan->update([
            'name' => $request->name,
            'min_amount' => $request->min_amount,
            'max_amount' => $request->max_amount,
            'days' => $request->days,
            'profit_percent' => $request->profit_percent,
        ]);

        return back()->with('success','Plan updated successfully!');
    }

    // Delete plan
    public function deletePlan($id)
    {
        $plan = WealthPlan::findOrFail($id);
        $plan->delete();
        return back()->with('success','Plan deleted successfully!');
    }

    // Show all user investments
    public function userInvestments()
    {
        $investments = UserWealth::with('user','plan')->orderBy('created_at','desc')->get();
        return view('admin.wealth.investments', compact('investments'));
    }

    // Update investment status manually
    public function updateInvestmentStatus(Request $request, $id)
    {
        $request->validate([
            'status'=>'required|in:0,1'
        ]);

        $inv = UserWealth::findOrFail($id);
        $inv->status = $request->status; // 1=active, 0=completed
        $inv->save();

        // If marking as completed manually, return invested amount + profit to user
        if($request->status == 0){
            $user = User::find($inv->user_id);
            $profit = ($inv->amount * $inv->plan->profit_percent)/100;
            $user->balance += $inv->amount + $profit;
            $user->save();
        }

        return back()->with('success','Investment status updated!');
    }
}
