<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Task;
use App\Models\UserTask;
use App\Models\UserLedger;
use App\Models\User;
use App\Models\Setting;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class UserTaskController extends Controller
{
    // Show all tasks for active packages
    public function index()
    {
        $user = auth()->user();
        $setting = Setting::first();

        if ($setting && $setting->is_weekend_break) {
            return redirect()->route('tas')
                ->with('error', 'Weekend Break is active!.');
        }

        $this->resetDailyTasks($user->id);

        $activePackageIds = DB::table('user_packages')
            ->where('user_id', $user->id)
            ->where('status', 'active')
            ->pluck('package_id');

        if ($activePackageIds->isEmpty()) {
            return redirect()->route('tas')
                ->with('error', 'You have no package please active first package.');
        }

        $tasks = Task::whereIn('package_id', $activePackageIds)->get();
        $userTasks = UserTask::where('user_id', $user->id)
            ->whereDate('completed_at', Carbon::today())
            ->pluck('task_id')
            ->toArray();

        return view('user.tasks.index', compact('tasks', 'userTasks'));
    }

    // Complete a task
    public function complete(Request $request, $taskId)
    {
        $user = auth()->user();
        $setting = Setting::first();

        if ($setting && $setting->is_weekend_break) {
            return back()->with('error', 'Weekend Break is active!.');
        }

        $activePackageIds = DB::table('user_packages')
            ->where('user_id', $user->id)
            ->where('status', 'active')
            ->pluck('package_id');

        if ($activePackageIds->isEmpty()) {
            return back()->with('error', 'You have no active package.');
        }

        $task = Task::findOrFail($taskId);

        if (!$activePackageIds->contains($task->package_id)) {
            return back()->with('error', 'This task is not in your plan.');
        }

        $alreadyCompleted = UserTask::where('user_id', $user->id)
            ->where('task_id', $task->id)
            ->whereDate('completed_at', Carbon::today())
            ->exists();

        if ($alreadyCompleted) {
            return back()->with('error', 'Task already completed.');
        }

        // Save task completion
        $userTask = new UserTask();
        $userTask->user_id = $user->id;
        $userTask->task_id = $task->id;
        $userTask->status = 'completed';
        $userTask->earned = $task->reward_amount;
        $userTask->completed_at = now();
        $userTask->save();

        // Add reward amount to user balance
        $user->increment('balance', $task->reward_amount);

        // Distribute task rebate commission + points to upline
        $this->distributeTaskRebateCommission($user, $task->reward_amount);

        return back()->with('success', 'Task completed!');
    }

    // Reset daily tasks for single user
    private function resetDailyTasks($userId)
    {
        $lastReset = UserTask::where('user_id', $userId)
            ->orderBy('created_at', 'desc')
            ->value('created_at');

        if (!$lastReset || Carbon::parse($lastReset)->lt(Carbon::today())) {
            UserTask::where('user_id', $userId)
                ->whereDate('completed_at', '<', Carbon::today())
                ->delete();
        }
    }

    // Tas page (summary)
    public function tas()
    {
        $user = auth()->user();
        $today = Carbon::today();

        $todayEarning = UserTask::where('user_id', $user->id)
            ->where('status', 'completed')
            ->whereDate('completed_at', $today)
            ->sum('earned');

        $completedTasks = UserTask::where('user_id', $user->id)
            ->where('status', 'completed')
            ->whereDate('completed_at', $today)
            ->count();

        $availableBalance = $user->balance;

        return view('app.main.tas', compact('todayEarning', 'completedTasks', 'availableBalance'));
    }

    // Distribute task rebate commission
    private function distributeTaskRebateCommission($user, $taskReward)
    {
        // Check if user has only free packages
        $activePackages = DB::table('user_packages')
            ->where('user_id', $user->id)
            ->where('status', 'active')
            ->pluck('package_id')
            ->toArray();

        $paidPackageIds = DB::table('packages')
            ->whereIn('id', $activePackages)
            ->where('type', '!=', 'free') // only paid packages
            ->pluck('id')
            ->toArray();

        if (empty($paidPackageIds)) {
            // User has only free packages, rebate commission & points disabled
            return;
        }

        // Only for paid package users
        $levels = [
            'first' => ['percent' => setting('task_rebate_first_level', 5), 'points' => setting('task_rebate_first_level_points', 2)],
            'second' => ['percent' => setting('task_rebate_second_level', 3), 'points' => setting('task_rebate_second_level_points', 1)],
            'third' => ['percent' => setting('task_rebate_third_level', 1), 'points' => setting('task_rebate_third_level_points', 1)],
        ];

        $first_ref = User::where('ref_id', $user->ref_by)->first();
        if ($first_ref) {
            $amount = $taskReward * $levels['first']['percent'] / 100;
            $first_ref->increment('balance', $amount);  
            $first_ref->increment('credit_score', $levels['first']['points']); 
            $this->createLedger($first_ref->id, $user->id, $amount, $levels['first']['points'], 'first_task_rebate');

            $second_ref = User::where('ref_id', $first_ref->ref_by)->first();
            if ($second_ref) {
                $amount = $taskReward * $levels['second']['percent'] / 100;
                $second_ref->increment('balance', $amount);
                $second_ref->increment('credit_score', $levels['second']['points']);
                $this->createLedger($second_ref->id, $user->id, $amount, $levels['second']['points'], 'second_task_rebate');

                $third_ref = User::where('ref_id', $second_ref->ref_by)->first();
                if ($third_ref) {
                    $amount = $taskReward * $levels['third']['percent'] / 100;
                    $third_ref->increment('balance', $amount);
                    $third_ref->increment('credit_score', $levels['third']['points']);
                    $this->createLedger($third_ref->id, $user->id, $amount, $levels['third']['points'], 'third_task_rebate');
                }
            }
        }
    }

    // Create ledger entry for task rebate
    private function createLedger($userId, $fromUserId, $amount, $points, $level)
    {
        $ledger = new UserLedger();
        $ledger->user_id = $userId;
        $ledger->get_balance_from_user_id = $fromUserId;
        $ledger->reason = 'task_refer_bonus';
        $ledger->perticulation = 'Task completed by referral';
        $ledger->amount = $amount;
        $ledger->credit = $amount;
        $ledger->task_rebate_points = $points; // for tracking only
        $ledger->status = 'approved';
        $ledger->step = $level;
        $ledger->date = now();
        $ledger->save();
    }

    /**
     * Public cron method - resets tasks for all users
     */
    public function resetDailyTasksCron()
    {
        $users = User::all();

        foreach ($users as $user) {
            $this->resetDailyTasks($user->id);
        }

        \Log::info("Daily tasks reset completed at " . now());

        return response()->json(['status' => 'success', 'message' => 'Daily tasks reset completed']);
    }
}
