<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\BankAccount;
use App\Models\UserLedger;
use App\Models\Withdrawal;
use App\Models\Purchase;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class WithdrawController extends Controller
{
    /**
     * Withdraw page
     */
    public function withdraw()
    {
        $banks = auth()->user()->bankAccounts ?? collect([]);

        if ($banks->count() < 1) {
            return redirect('card')->with('error', 'Please add a bank account first.');
        }

        return view('app.main.withdraw.index', compact('banks'));
    }

    /**
     * Withdraw request
     */
    public function withdrawRequest(Request $request)
    {
        // Validate request including login password
        $validate = Validator::make($request->all(), [
            'amount'   => 'required|numeric',
            'bank_id'  => 'required|exists:bank_accounts,id',
            'password' => 'required|string',
        ]);

        if ($validate->fails()) {
            return redirect()->back()->withErrors($validate)->withInput();
        }

        $user = Auth::user();
        $startTime = '10:00';
        $endTime = '20:00';
        $currentTime = now()->format('H:i');

        if ($currentTime < $startTime || $currentTime > $endTime) {
        return redirect()->back()->with('error', 'Withdrawals are allowed only between 10 AM and 7 PM.');
        }

        $todayWithdrawal = Withdrawal::where('user_id', $user->id)
            ->whereDate('created_at', now()->toDateString())
            ->first();

        if ($todayWithdrawal) {
            return redirect()->back()->with('error', 'You can only make one withdrawal per day.');
        }
        // ✅ Check login password
        if (!Hash::check($request->password, $user->password)) {
            return redirect()->back()->with('error', 'Fund password is incorrect.');
        }

        // ✅ Credit score check
        if ($user->credit_score < 10) {
            return redirect()->back()->with('error', 'Cannot withdraw, your points are less than 10.');
        }

    
        // ✅ Balance check
        if ($request->amount > $user->balance) {
            return redirect()->back()->with('error', 'Insufficient balance.');
        }

        // ✅ Withdraw limits
        if ($request->amount < setting('minimum_withdraw')) {
            return redirect()->back()->with('error', 'Amount less than minimum withdraw limit.');
        }

        if ($request->amount > setting('maximum_withdraw')) {
            return redirect()->back()->with('error', 'Amount greater than maximum withdraw limit.');
        }

        // ✅ Charge calculation
        $charge = 0;
        if (setting('withdraw_charge') > 0) {
            $charge = ($request->amount * setting('withdraw_charge')) / 100;
        }

        // ✅ Bank check
        $bank = BankAccount::where('id', $request->bank_id)
            ->where('user_id', $user->id)
            ->first();

        if (!$bank) {
            return redirect()->back()->with('error', 'Invalid bank selected.');
        }

        // ✅ Update balance
        $user->balance -= $request->amount;
        $user->save();

        // ✅ Withdrawal record
        $withdrawal = new Withdrawal();
        $withdrawal->user_id      = $user->id;
        $withdrawal->method_name  = $bank->bank_name;
        $withdrawal->number       = $bank->account_number;
        $withdrawal->amount       = $request->amount;
        $withdrawal->currency     = 'Kenya';
        $withdrawal->charge       = $charge;
        $withdrawal->oid          = 'W-' . rand(100000, 999999) . rand(100000, 999999);
        $withdrawal->final_amount = $request->amount - $charge;
        $withdrawal->status       = 'pending';
        $withdrawal->save();

        // ✅ Ledger entry
        $ledger = new UserLedger();
        $ledger->user_id       = $user->id;
        $ledger->reason        = 'withdraw_request';
        $ledger->perticulation = 'Withdraw request status is pending';
        $ledger->amount        = $request->amount;
        $ledger->debit         = $request->amount - $charge;
        $ledger->status        = 'pending';
        $ledger->date          = now()->format('d-m-Y H:i');
        $ledger->save();

        return redirect()->back()->with('success', 'Withdraw request submitted successfully.');
    }

    /**
     * Withdraw Preview
     */
    public function withdrawPreview()
    {
        $withdraws = Withdrawal::where('user_id', Auth::id())
            ->orderByDesc('id')
            ->get();

        return view('app.main.withdraw.withdraw-preview', compact('withdraws'));
    }
}
